/****************************************************************************
 *  Catfood Software RLIB.DLL - DLList Module                               *
 *                                                                          *
 *  Double linked list functions - written for multiple level undo/redo so  *
 *  make sure it does everthing you need before you start using it!         *
 ****************************************************************************
 *  (c)Copyright 1997-1998 Catfood Software. This software may be used      *
 *  without charge provided that Catfood Software is credited in your about *
 *  box or splash screen including the Catfood URL: http://www.catfood.net/ *
 *  While every effort has been made to ensure that RLIB is free from       *
 *  defects you may not use this software without accepting that Catfood    *
 *  Software is indemnified from all liabilities resulting from the use of  *
 *  RLIB.                                                                   *
 ****************************************************************************
 *  Ver   Date        Who  Description                                      *
 *  1.00  29-06-1998  RDE  First RLIB version                               *
 ****************************************************************************/

#ifndef _DLLIST_H
#define _DLLIST_H

#ifdef __cplusplus
extern "C" {            // C declarations for C++
#endif	// __cplusplus 

/****************************************************************************
 *  Function     Summary                                     Ordinal        *
 ****************************************************************************
 *  DllInit      Create a linked list                        @2             *
 *  DllGetLen    Get the number of items in the list         @3             *
 *  DllAppend    Add an item to the list                     @4             *
 *  DllGetDPAt   Get a pointer to an item from the list      @5             *
 *  DllFreeDPAt  Free a pointer to an item from the list     @6             *
 *  DllTruncate  Remove the last n items in the list         @7             *
 *  DllKill      Destroy the list                            @8             *
 ****************************************************************************/


/****************************************************************************
 *  DllInit                                                                 *
 ****************************************************************************
 *  Description:  Create a linked list                                      *
 *                                                                          *
 *  Parameters:   ppList  Pointer to a pointer that will be used to refer   *
 *                        to the list once it has been created              *
 *                                                                          *
 *  Returns:      BOOL    TRUE for success                                  *
 *                        FALSE for failure                                 *
 ****************************************************************************/

BOOL DllInit(void** ppList);

/****************************************************************************
 *  DllGetLen                                                               *
 ****************************************************************************
 *  Description:  Get the length of the list                                *
 *                                                                          *
 *  Parameters:   pList  List pointer                                       *
 *                                                                          *
 *  Returns:      int    Number of items in the list                        *
 ****************************************************************************/

int DllGetLen(void* pList);

/****************************************************************************
 *  DllAppend                                                               *
 ****************************************************************************
 *  Description:  Add an item to the list. The item is copied so you are    *
 *                responsible for dealing with the original data            *
 *                                                                          *
 *  Parameters:   pList     List pointer                                    *
 *                pData     Pointer to data to add to the list              *
 *                nDataLen  Length of data                                  *
 *                                                                          *
 *  Returns:      BOOL      TRUE for success                                *
 *                          FALSE for failure                               *
 ****************************************************************************/

BOOL DllAppend(void* pList, void* pData, int nDataLen);

/****************************************************************************
 *  DllGetDPAt                                                              *
 ****************************************************************************
 *  Description:  Get a pointer to the data at an index in the list. This   *
 *                pointer must be released using DllFreeDPAt when you are   *
 *                finished with it.                                         *
 *                                                                          *
 *  Parameters:   pList   List pointer                                      *
 *                ppData  Pointer to the pointer to set to the data address *
 *                nIndex  Index of the list item                            *
 *                                                                          *
 *  Returns:      BOOL    TRUE for success                                  *
 *                        FALSE for failure                                 *
 ****************************************************************************/

BOOL DllGetDPAt(void* pList, void** ppData, int nIndex);

/****************************************************************************
 *  DllFreeDPAt                                                             *
 ****************************************************************************
 *  Description:  Free a pointer obtained in a previous call to DLLGetDPAt  *
 *                                                                          *
 *  Parameters:   pList   List pointer                                      *
 *                nIndex  Index of the list item                            *
 *                                                                          *
 *  Returns:      BOOL    TRUE for success                                  *
 *                        FALSE for failure                                 *
 ****************************************************************************/

BOOL DllFreeDPAt(void* pList, int nIndex);

/****************************************************************************
 *  DllTruncate                                                             *
 ****************************************************************************
 *  Description:  Truncate the list                                         *
 *                                                                          *
 *  Parameters:   pList        List pointer                                 *
 *                nFirstToDie  The index of first item to remove - i.e.     *
 *                             this item and all subsequent items will be   *
 *                             removed from the list                        *
 *                                                                          *
 *  Returns:      BOOL         TRUE for success                             *
 *                             FALSE for failure                            *
 ****************************************************************************/

BOOL DllTruncate(void* pList, int nFirstToDie);

/****************************************************************************
 *  DllKill                                                                 *
 ****************************************************************************
 *  Description:  Destroy the list, including any data items                *
 *                                                                          *
 *  Parameters:   pList  List pointer                                       *
 *                                                                          *
 *  Returns:      BOOL   TRUE for success                                   *
 *                       FALSE for failure                                  *
 ****************************************************************************/

BOOL DllKill(void* pList);



#ifdef __cplusplus
}
#endif	// __cplusplus

#endif // _DLLIST_H

